<?php
/* --------------------------------------------------------------
   WithdrawalsController.inc.php 2020-04-21
   Gambio GmbH
   http://www.gambio.de
   Copyright © 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

use Gambio\Admin\Withdrawal\Interfaces\Withdrawal;
use Gambio\Admin\Withdrawal\Interfaces\WithdrawalService;
use Gambio\Admin\Withdrawal\Models\OrderId;
use Gambio\Admin\Withdrawal\Models\WithdrawalId;
use Gambio\Admin\Withdrawal\SqlCriteria\WithdrawalFilters;
use Gambio\Admin\Withdrawal\SqlCriteria\WithdrawalSorting;
use Gambio\Admin\Withdrawal\SqlCriteria\WithdrawalSortings;
use Gambio\Admin\Withdrawal\SqlCriteria\WithdrawalSqlCriteria;
use Gambio\Admin\Withdrawal\SqlCriteria\WithdrawalSqlPagination;

/**
 * Class WithdrawalsController
 */
class WithdrawalsController extends AdminHttpViewController
{
    /**
     * @var WithdrawalService
     */
    protected $service;
    
    
    /**
     * Initialization of VPE controller
     */
    public function init()
    {
        $this->service = LegacyDependencyContainer::getInstance()->get(WithdrawalService::class);
    }
    
    
    /**
     * @return AdminLayoutHttpControllerResponse
     */
    public function actionDefault(): AdminLayoutHttpControllerResponse
    {
        $languageTextManager = MainFactory::create('LanguageTextManager', 'withdrawals', $_SESSION['languages_id']);
        
        $title = new NonEmptyStringType($languageTextManager->get_text('title'));
        
        $template = new ExistingFile(new NonEmptyStringType(DIR_FS_ADMIN . '/html/content/withdrawals/overview.html'));
        
        $assets = MainFactory::create('AssetCollection');
        $assets->add(MainFactory::create('Asset', 'admin_buttons.lang.inc.php'));
        $assets->add(MainFactory::create('Asset', 'withdrawals.lang.inc.php'));
        
        return MainFactory::create('AdminLayoutHttpControllerResponse', $title, $template, null, $assets);
    }
    
    
    /**
     * @return JsonHttpControllerResponse
     */
    public function actionGetWithdrawals(): JsonHttpControllerResponse
    {
        $_SESSION['coo_page_token']->is_valid($this->_getPostData('page_token'));
        
        try {
            $offset = (int)$this->_getPostData('offset');
            $offset = ($offset >= 0) ? $offset : 0;
            
            $limit = (int)$this->_getPostData('limit');
            $limit = ($limit >= 1) ? $limit : 10;
            
            $filters = WithdrawalFilters::create();
            
            $sortBy    = $this->_getPostData('sortBy') ?? 'id';
            $sortOrder = $this->_getPostData('sortOrder') ?? 'id';
            $sorting   = WithdrawalSortings::create(new WithdrawalSorting($sortBy, $sortOrder));
            
            $criteria   = WithdrawalSqlCriteria::create($filters, $sorting);
            $pagination = WithdrawalSqlPagination::create($limit, $offset);
            
            $data = [];
            /** @var Withdrawal $withdrawal */
            foreach ($this->service->getAllWithdrawals($criteria, $pagination) as $withdrawal) {
                $data[] = [
                    'id'         => $withdrawal->id(),
                    'date'       => $withdrawal->date(DATE_FORMAT),
                    'customer'   => $withdrawal->customerFirstName() . ' ' . $withdrawal->customerLastName(),
                    'customerId' => $withdrawal->customerId(),
                    'orderId'    => $withdrawal->orderId(),
                ];
            }
            $total = $this->service->getTotalCountOfWithdrawals($criteria);
            
            return MainFactory::create('JsonHttpControllerResponse',
                                       ['success' => true, 'withdrawals' => $data, 'total' => $total]);
        } catch (Exception $exception) {
            return MainFactory::create('JsonHttpControllerResponse',
                                       [
                                           'success'     => false,
                                           'withdrawals' => [],
                                           'total'       => 0,
                                           'error'       => $exception->getMessage()
                                       ]);
        }
    }
    
    
    /**
     * @return JsonHttpControllerResponse
     */
    public function actionGetWithdrawal(): JsonHttpControllerResponse
    {
        $_SESSION['coo_page_token']->is_valid($this->_getPostData('page_token'));
        
        try {
            $id         = (int)$this->_getPostData('id');
            $withdrawal = $this->service->getWithdrawalById(WithdrawalId::create($id));
            $data       = [
                'id'             => $withdrawal->id(),
                'order'          => [
                    'id'           => $withdrawal->orderId(),
                    'creationDate' => ($withdrawal->orderCreationDate() !== null) ? date(DATE_FORMAT,
                                                                                         strtotime($withdrawal->orderCreationDate())) : '',
                    'deliveryDate' => ($withdrawal->orderDeliveryDate() !== null) ? date(DATE_FORMAT,
                                                                                         strtotime($withdrawal->orderDeliveryDate())) : '',
                ],
                'customer'       => [
                    'id'        => $withdrawal->customerId(),
                    'gender'    => $withdrawal->customerGender(),
                    'firstName' => $withdrawal->customerFirstName(),
                    'lastName'  => $withdrawal->customerLastName(),
                    'address'   => [
                        'street'   => $withdrawal->customerStreet(),
                        'postcode' => $withdrawal->customerPostcode(),
                        'city'     => $withdrawal->customerCity(),
                        'country'  => $withdrawal->customerCountry(),
                    ],
                    'email'     => $withdrawal->customerEmail(),
                ],
                'date'           => $withdrawal->date(DATE_FORMAT),
                'content'        => $withdrawal->content(),
                'createdByAdmin' => $withdrawal->wasCreatedByAdmin(),
                'createdOn'      => $withdrawal->createdOn(DATE_FORMAT),
            ];
            
            return MainFactory::create('JsonHttpControllerResponse', ['success' => true, 'withdrawal' => $data]);
        } catch (Exception $exception) {
            return MainFactory::create('JsonHttpControllerResponse',
                                       ['success' => false, 'error' => $exception->getMessage()]);
        }
    }
    
    
    /**
     * @return JsonHttpControllerResponse
     */
    public function actionChangeOrderId(): JsonHttpControllerResponse
    {
        $_SESSION['coo_page_token']->is_valid($this->_getPostData('page_token'));
        
        try {
            $id         = (int)$this->_getPostData('id');
            $orderId    = (int)$this->_getPostData('orderId');
            $withdrawal = $this->service->getWithdrawalById(WithdrawalId::create($id));
            
            $withdrawal->changeOrder(OrderId::create($orderId));
            $this->service->storeWithdrawal($withdrawal);
            
            return MainFactory::create('JsonHttpControllerResponse', ['success' => true]);
        } catch (Exception $exception) {
            return MainFactory::create('JsonHttpControllerResponse',
                                       ['success' => false, 'error' => $exception->getMessage()]);
        }
    }
    
    
    /**
     * @return JsonHttpControllerResponse
     */
    public function actionDeleteWithdrawal(): JsonHttpControllerResponse
    {
        $_SESSION['coo_page_token']->is_valid($this->_getPostData('page_token'));
        
        try {
            $id = (int)$this->_getPostData('id');
            $this->service->deleteWithdrawal(WithdrawalId::create($id));
            
            return MainFactory::create('JsonHttpControllerResponse', ['success' => true]);
        } catch (Exception $exception) {
            return MainFactory::create('JsonHttpControllerResponse',
                                       ['success' => false, 'error' => $exception->getMessage()]);
        }
    }
}